/******************************************************************************
 *
 *  fs_scratch.c - Defines FsScratchSpace
 *
 *  Copyright (C) 2009 Monotype Imaging Inc. All rights reserved.
 *
 *  Confidential information of Monotype Imaging Inc.
 *
 *****************************************************************************/


#include "fs_scratch_priv.h"
#include "fs_scratch.h"
#include "fs_function.h"


typedef union
{
    void *vp;
    struct
    {
        FS_ULONG size;
        FS_BYTE  flag;
    } sf;
} SSUnion;

#define SS_FREE 0
#define SS_USED 0x10

#define SSOHEAD (sizeof(SSUnion))
#define SSGRAIN (SSOHEAD - 1)

FsScratchSpace *
FsScratchSpace_new(FS_STATE *FS_state_ptr, FS_ULONG size)
{
    FsScratchSpace *ss = FSS_malloc(FS_state_ptr, sizeof(FsScratchSpace));
    if (!ss)
    {
        return NULL;
    }
    else
    {
        FS_LONG err = FsScratchSpace_init(ss, FS_state_ptr, size);
        if (err != SUCCESS)
        {
            FSS_free(FS_state_ptr, ss);
            ss = NULL;
        }
        return ss;
    }
}

FS_LONG
FsScratchSpace_init(FsScratchSpace *ss, FS_STATE *FS_state_ptr, FS_ULONG size)
{
    size = (size + SSGRAIN) & ~SSGRAIN;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FsScratchSpace buffer";
#endif
    ss->buffer = FSS_malloc(FS_state_ptr, size);
    if (ss->buffer)
    {
        SSUnion *ssu = (SSUnion *)ss->buffer;
        ssu->vp = NULL;  /* not really needed, but gets rid of PC-Lint warning */
        ssu->sf.size = size - SSOHEAD;
        ssu->sf.flag = SS_FREE;

        ss->beyond = (FS_BYTE *)ss->buffer + size;

    }
    else ss->beyond = NULL;

    return FS_state_ptr->error;
}

void
FsScratchSpace_done(FsScratchSpace *ss, FS_STATE *FS_state_ptr)
{
    if (ss->buffer)
    {
#ifdef FS_DEBUG
        SSUnion *ssu;
#endif

#ifdef FS_DEBUG
        ssu = (SSUnion *)ss->buffer;
        /* ssu->sf.size + SSOHEAD should equal initial size if all space released properly */
        /* FS_PRINTF(("%% final scratch space size =%9lu\n",ssu->sf.size+SSOHEAD));        */
        if ((FS_ULONG)(ss->beyond - ss->buffer) != (ssu->sf.size + SSOHEAD))
            FS_state_ptr->error = TRASHED_MEM_ERR;
#endif
        FSS_free(FS_state_ptr, ss->buffer);
        ss->buffer = NULL;
        ss->beyond = NULL;
    }
}

void
FsScratchSpace_delete(FsScratchSpace *ss, FS_STATE *FS_state_ptr)
{
    FsScratchSpace_done(ss, FS_state_ptr);
    FSS_free(FS_state_ptr, ss);
}

void *
FsScratchSpace_reserve(FsScratchSpace *ss,
                       FS_STATE *FS_state_ptr, FS_ULONG nbytes)
{
    FS_ULONG orig = nbytes;
    FS_BYTE *p = (FS_BYTE *)(ss->buffer);
    FS_BYTE *beyond = (FS_BYTE *)(ss->beyond);

    nbytes = (nbytes + SSGRAIN) & ~SSGRAIN;

    do
    {
        SSUnion *ssu = (SSUnion *)p;

        if (!(ssu->sf.flag & SS_USED) && (ssu->sf.size >= nbytes))
        {
            ssu->sf.flag = SS_USED;

            if ((ssu->sf.size - nbytes) > SSOHEAD)
            {
                FS_BYTE *q;
                FS_ULONG remain = ssu->sf.size - nbytes;

                ssu->sf.size = nbytes;

                q = p + (ssu->sf.size + SSOHEAD);

                ssu = (SSUnion *)q;

                ssu->sf.size = remain - SSOHEAD;
                ssu->sf.flag = SS_FREE;
            }
            FS_state_ptr->error = SUCCESS;
            return p + SSOHEAD;
        }
        p += (ssu->sf.size + SSOHEAD);

    }
    while (p < beyond);

    p = FSS_malloc(FS_state_ptr, orig);

    return p;
}

FS_LONG
FsScratchSpace_release(FsScratchSpace *ss,
                       FS_STATE *FS_state_ptr, void *addr)
{
    if (((FS_BYTE *)addr < (FS_BYTE *)(ss->buffer)) ||
        ((FS_BYTE *)addr >= (FS_BYTE *)(ss->beyond)))
    {
        FSS_free(FS_state_ptr, addr);
        return FS_state_ptr->error = SUCCESS;
    }
    else
    {
        FS_BYTE *prev = NULL, *p = (FS_BYTE *)(ss->buffer);
        FS_BYTE *beyond = (FS_BYTE *)(ss->beyond);

        do
        {
            if ((p + SSOHEAD) == addr)
            {
                SSUnion *ssu = (SSUnion *)p;

                if (!(ssu->sf.flag & SS_USED))
                {
                    return FS_state_ptr->error = ERR_FREE_BAD_PTR;
                }
                ssu->sf.flag = SS_FREE;

                if (prev)
                {
                    SSUnion *prev_ssu = (SSUnion *)prev;

                    if (!(prev_ssu->sf.flag & SS_USED))
                    {
                        prev_ssu->sf.size += (ssu->sf.size + SSOHEAD);
                        p = prev;
                        ssu = prev_ssu;
                    }
                }
                if ((p + (ssu->sf.size + SSOHEAD)) < beyond)
                {
                    FS_BYTE *next = p + (ssu->sf.size + SSOHEAD);
                    SSUnion *next_ssu = (SSUnion *)next;

                    if (!(next_ssu->sf.flag & SS_USED))
                    {
                        ssu->sf.size += (next_ssu->sf.size + SSOHEAD);
                    }
                }
                return FS_state_ptr->error = SUCCESS;
            }
            else
            {
                SSUnion *ssu = (SSUnion *)p;

                prev = p;
                p += (ssu->sf.size + SSOHEAD);
            }
        }
        while (p < beyond);

        return FS_state_ptr->error = ERR_FREE_BAD_PTR;
    }
}

FS_LONG
FsScratchSpace_shrink(FsScratchSpace *ss, FS_STATE *FS_state_ptr, int *gotSome)
{
    (void)ss;  /* unused for now */
    *gotSome = 0;
    return FS_state_ptr->error = SUCCESS;
}
